# authentic2 - versatile identity manager
# Copyright (C) 2010-2019 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging

import ldap
from django.utils.encoding import force_str
from ldap.dn import escape_dn_chars
from ldap.filter import filter_format

from authentic2.backends.ldap_backend import LDAPBackend, UserCreationError, ldap_error_str
from authentic2.base_commands import LogToConsoleCommand

log = logging.getLogger("authentic2.backends.ldap_backend")


class Command(LogToConsoleCommand):
    loggername = "authentic2.backends.ldap_backend"

    def add_arguments(self, parser):
        parser.add_argument("username", help="Specify username to synchronize")

    def core_command(self, *args, **kwargs):
        username = kwargs["username"]
        log.info("Synchronize user %s...", username)
        backend = LDAPBackend()
        for block in backend.get_config():
            if block["authentication"] is False:
                continue
            if "%s" not in block["user_filter"]:
                log.error("User filter doesn't contain '%s'")
                continue
            for conn in backend.get_connections(block):
                ldap_uri = conn.get_option(ldap.OPT_URI)
                authz_ids = []
                user_basedn = force_str(block.get("user_basedn") or block["basedn"])
                try:
                    if block["user_dn_template"]:
                        template = force_str(block["user_dn_template"])
                        escaped_username = escape_dn_chars(username)
                        authz_ids.append(template.format(username=escaped_username))
                    else:
                        try:
                            if block["user_filter"]:
                                # allow multiple occurrences of the username in the filter
                                user_filter = force_str(block["user_filter"])
                                n = len(user_filter.split("%s")) - 1
                                try:
                                    query = filter_format(user_filter, (username,) * n)
                                except TypeError as e:
                                    log.error(
                                        "[%s] user_filter syntax error %r: %s",
                                        ldap_uri,
                                        block["user_filter"],
                                        e,
                                    )
                                    continue
                                log.debug(
                                    "[%s] looking up dn for username %r using query %r",
                                    ldap_uri,
                                    username,
                                    query,
                                )
                                results = conn.search_s(
                                    user_basedn, ldap.SCOPE_SUBTREE, query, ["1.1"]
                                )
                                results = backend.normalize_ldap_results(results)
                                # remove search references
                                results = [result for result in results if result[0] is not None]
                                log.debug("found dns %r", results)
                                if len(results) == 0:
                                    log.debug(
                                        "[%s] user lookup failed: no entry found, %s",
                                        ldap_uri,
                                        query,
                                    )
                                elif not block["multimatch"] and len(results) > 1:
                                    log.error(
                                        "[%s] user lookup failed: too many (%d) entries found: %s",
                                        ldap_uri,
                                        len(results),
                                        query,
                                    )
                                else:
                                    authz_ids.extend(result[0] for result in results)
                            else:
                                log.warning("[%s] no user_filter configured", ldap_uri)
                                continue
                        except ldap.NO_SUCH_OBJECT:
                            log.error(
                                "[%s] user lookup failed, basedn %s not found",
                                ldap_uri,
                                user_basedn,
                            )
                            if block["replicas"]:
                                break
                            continue
                        except ldap.LDAPError as e:
                            log.error(
                                "[%s] user lookup failed, with query %r (%s)",
                                ldap_uri,
                                query,
                                ldap_error_str(e),
                            )
                            continue
                    for authz_id in authz_ids:
                        try:
                            backend.get_ldap_attributes(block, conn, authz_id)
                        except ldap.NO_SUCH_OBJECT:
                            log.error("[%s] user %s not found", ldap_uri, authz_id)
                            if block["replicas"]:
                                break

                        try:
                            backend._return_user(authz_id, None, conn, block)
                            log.info("[%s] user %s synchronized", ldap_uri, authz_id)
                        except UserCreationError:
                            log.error("[%s] user %s not found", ldap_uri, authz_id)
                except ldap.CONNECT_ERROR:
                    log.error(
                        "[%s] connection to %r failed, did you forget to declare the TLS "
                        "certificate in /etc/ldap/ldap.conf ?",
                        ldap_uri,
                        block["url"],
                    )
                except ldap.TIMEOUT:
                    log.error("connection to %r timed out", block["url"])
                except ldap.SERVER_DOWN:
                    log.error("ldap authentication error: %r is down", block["url"])
                finally:
                    del conn


# vim: tabstop=4 shiftwidth=4 softtabstop=4 expandtab
