# authentic2 - versatile identity manager
# vim: shiftwidth=4 tabstop=4 expandtab
# Copyright (C) 2010-2020 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import logging
from datetime import timedelta

from django.core.management.base import BaseCommand
from django.utils import timezone

from authentic2.backends import get_user_queryset

logger = logging.getLogger(__name__)


class Command(BaseCommand):
    help = """Delete disabled accounts"""

    verbosity_to_log_level = {
        0: logging.CRITICAL,
        1: logging.WARNING,
        2: logging.INFO,
        3: logging.DEBUG,
    }

    def add_arguments(self, parser):
        parser.add_argument(
            "-D",
            "--delay",
            type=int,
            help="Delay after deactivation in days before deletion (default: 30 days)",
            default=30,
        )
        parser.add_argument(
            "-R",
            "--reason",
            help="Focus on disabled users for specified reason (example: ldap-not-present)",
        )
        parser.add_argument(
            "-M",
            "--max-count",
            type=int,
            help="Max count of users to delete",
        )
        parser.add_argument("--fake", action="store_true", help="do nothing", default=False)

    def handle(self, *args, **options):
        self.delay = options["delay"]
        self.fake = options["fake"]

        # add StreamHandler for console output
        handler = logging.StreamHandler()
        handler.setLevel(level=self.verbosity_to_log_level[options["verbosity"]])
        logger.addHandler(handler)
        # prevent logging to external logs when fake
        if self.fake:
            logger.propagate = False

        try:
            limit = timezone.now() - timedelta(days=self.delay)
            logger.info("Looking for users disabled since at least %s", limit)
            filter_kwargs = {
                "deactivation__lte": limit,
                "is_active": False,
            }
            if options["reason"]:
                logger.info("Filter on users with deactivation reason '%s'", options["reason"])
                filter_kwargs["deactivation_reason"] = options["reason"]
            users = get_user_queryset().filter(**filter_kwargs)
            logger.info("%d users to delete", users.count())
            count = 0
            for user in users:
                logger.info(
                    "Deleting user %s (disabled since %s, reason=%s)",
                    user,
                    user.deactivation,
                    user.deactivation_reason,
                )
                if not self.fake:
                    user.delete()
                else:
                    logger.debug("Fake mode: delete user %s", user)
                count += 1
                if options["max_count"] and count >= options["max_count"]:
                    logger.info(
                        "Max users to delete limit reached (%d), stop", options["max_count"]
                    )
                    break
            logger.info("%d users deleted", count)
        except Exception:
            logger.exception("delete-disabled-accounts failed")
