import type { IMessage, ISubscription, IRoom } from '@rocket.chat/core-typings';
import { useMethod, usePermission, useSetting, useUser } from '@rocket.chat/ui-contexts';
import { useMemo } from 'react';

import { AutoTranslate } from '../../../../app/autotranslate/client';
import type { MessageActionConfig } from '../../../../app/ui-utils/client/lib/MessageAction';
import { roomCoordinator } from '../../../lib/rooms/roomCoordinator';
import { Messages } from '../../../stores';
import { hasTranslationLanguageInAttachments, hasTranslationLanguageInMessage } from '../../../views/room/MessageList/lib/autoTranslate';

export const useTranslateAction = (
	message: IMessage & { autoTranslateShowInverse?: boolean },
	{ room, subscription }: { room: IRoom; subscription: ISubscription | undefined },
): MessageActionConfig | null => {
	const user = useUser();
	const autoTranslateEnabled = useSetting('AutoTranslate_Enabled', false);
	const canAutoTranslate = usePermission('auto-translate');
	const translateMessage = useMethod('autoTranslate.translateMessage');

	const language = useMemo(
		() => subscription?.autoTranslateLanguage || AutoTranslate.getLanguage(message.rid),
		[message.rid, subscription?.autoTranslateLanguage],
	);
	const hasTranslations = useMemo(
		() => hasTranslationLanguageInMessage(message, language) || hasTranslationLanguageInAttachments(message.attachments, language),
		[message, language],
	);

	const updateMessages = Messages.use((state) => state.update);

	if (!autoTranslateEnabled || !canAutoTranslate || !user) {
		return null;
	}

	const isLivechatRoom = roomCoordinator.isLivechatRoom(room?.t);
	const isDifferentUser = message?.u && message.u._id !== user._id;
	const autoTranslationActive = subscription?.autoTranslate || isLivechatRoom;

	if (!message.autoTranslateShowInverse && (!isDifferentUser || !autoTranslationActive || hasTranslations)) {
		return null;
	}

	return {
		id: 'translate',
		icon: 'language',
		label: 'Translate',
		context: ['message', 'message-mobile', 'threads'],
		type: 'interaction',
		group: 'menu',
		action() {
			if (!hasTranslations) {
				AutoTranslate.messageIdsToWait[message._id] = true;
				updateMessages(
					(record) => record._id === message._id,
					(record) => ({ ...record, autoTranslateFetching: true }),
				);
				void translateMessage(message, language);
			}

			updateMessages(
				(record) => record._id === message._id,
				'autoTranslateShowInverse' in message
					? ({ autoTranslateShowInverse: _, ...record }) => record
					: (record) => ({ ...record, autoTranslateShowInverse: true }),
			);
		},
		order: 90,
	};
};
