import json
import tomllib
from datetime import UTC, datetime
from os import environ
from pathlib import PosixPath
from time import time
from typing import Any

import werkzeug.exceptions
from flask import Flask, abort, request
from flask.logging import create_logger
from xapian import (DB_CREATE, DB_CREATE_OR_OPEN, DB_OPEN, Database,
                    DatabaseNotFoundError)
from xapian import Document as XapianDocument
from xapian import (Enquire, QueryParser, Stem, TermGenerator,
                    WritableDatabase, sortable_serialise, sortable_unserialise)

from eego import Document


class EegoWritableDatabase(WritableDatabase):
    def __init__(self) -> None:  # pylint: disable=super-init-not-called
        environ["XAPIAN_FLUSH_THRESHOLD"] = "100000"

        self.last_commit = time()
        self.lazy_initialized = False

    def lazy_init(self, app: Flask) -> None:
        if not self.lazy_initialized:
            super().__init__(
                str(app.config["DB_PATH"]),
                DB_CREATE_OR_OPEN,
            )
            self.lazy_initialized = True

    def commit_batch(self) -> None:
        if time() - self.last_commit > 300:
            self.commit()

    def commit(self) -> bool:
        if self.lazy_initialized:
            super().commit()
            self.last_commit = time()
            return True
        return False


# pylint: disable=too-many-statements
def create_app(
    wdb: EegoWritableDatabase,
) -> Flask:
    app = Flask(__name__)
    app.config["DB_PATH"] = PosixPath("/tmp/xapian_eego_db")
    log = create_logger(app)
    cfg_fname = PosixPath("/etc/eego.toml")
    if not cfg_fname.is_file():
        log.warning(f"Config file not found: {cfg_fname}. Applying default settings.")
    else:
        with cfg_fname.open("rb") as fin:
            cfg = tomllib.load(fin)
            app.config.from_mapping(cfg.get("xapian", {}))
    app.config["DB_PATH"] = PosixPath(app.config["DB_PATH"])

    # Make sure the xapian DB is initialized
    if not app.config["DB_PATH"].is_dir():
        log.warning(
            "Initialized empty Xapian database in %s", app.config["DB_PATH"].resolve()
        )
        app.config["DB_PATH"].mkdir(parents=True)
        WritableDatabase(str(app.config["DB_PATH"]), DB_CREATE)

    @app.route("/doc", methods=["GET"])
    def get_last() -> str:
        if request.args.get("sort") != "modtime-cache-descending":
            raise werkzeug.exceptions.NotImplemented
        source = request.args.get("source")
        if source is None:
            raise werkzeug.exceptions.BadRequestKeyError

        try:
            database = Database(str(app.config["DB_PATH"]), DB_OPEN)
        except DatabaseNotFoundError:
            log.exception(f"Xapian database not found: {app.config['DB_PATH']}")
            return str(0)
        queryparser = QueryParser()
        queryparser.add_prefix("source", "XS")
        query = queryparser.parse_query("source:" + source)
        enquire = Enquire(database)
        enquire.set_query(query)
        enquire.set_sort_by_value(6, True)
        results = enquire.get_mset(0, 1)
        if results.size() != 1:
            return str(0)
        return str(sortable_unserialise(results[0].document.get_value(6)))

    @app.route("/doc", methods=["POST"])
    def create_or_update() -> str:
        obj: dict[str, Any] | None = request.get_json()
        if obj is None:
            raise werkzeug.exceptions.BadRequestKeyError
        received = Document(**obj)
        doc = XapianDocument()

        doc.add_boolean_term("XS" + received.source)

        termgenerator = TermGenerator()
        termgenerator.set_stemmer(Stem("french"))
        termgenerator.set_document(doc)
        termgenerator.index_text(received.title.lower(), 1, "S")
        termgenerator.index_text(received.url, 1, "Q")
        termgenerator.index_text(received.content.lower())

        doc.set_data(
            "title="
            + received.title
            + "\n"
            + "url="
            + received.url
            + "\n"
            + "sample="
            + received.content.replace("\n", " ")
            + "\n"
            + "source="
            + received.source
            + "\n"
            + "size="
            + str(received.size)
            + "\n"
            + "modtime="
            + str(received.modtime)
        )
        doc.add_value(
            5,
            datetime.fromtimestamp(received.modtime, tz=UTC).strftime("%Y%m%d"),
        )
        doc.add_value(
            6,
            sortable_serialise(received.modtime_cache or received.modtime),
        )

        doc.add_boolean_term("Q" + received.url)
        wdb.lazy_init(app)
        wdb.replace_document("Q" + received.url, doc)
        wdb.commit_batch()
        return json.dumps({"msg": "Document successfully added"})

    @app.route("/doc/<path:url>", methods=["DELETE"])
    def delete(url: str) -> str:
        database = Database(str(app.config["DB_PATH"]), DB_OPEN)
        queryparser = QueryParser()
        queryparser.add_prefix("id", "Q")
        query = queryparser.parse_query("id:" + url)
        enquire = Enquire(database)
        enquire.set_query(query)
        results = enquire.get_mset(0, 1)
        if results.size() != 1:
            abort(404)

        wdb.lazy_init(app)
        wdb.delete_document("Q" + url)
        wdb.commit_batch()
        return json.dumps({"msg": "Document successfully deleted."})

    @app.route("/doc", methods=["PATCH"])
    def commit() -> str:
        if wdb.commit():
            return json.dumps({"msg": "Modification committed."})
        return json.dumps({"msg": "Nothing to commit."})

    return app
