from pathlib import PosixPath

from eego import Document, get_config
from eego.injector import DirectoryInjector, Injector


def test_injector_get_pdf_title_with_title() -> None:
    with PosixPath("tests/resources/with_title.pdf").open("rb") as file:
        assert Injector(get_config(test=True)).get_pdf_title(file) == "Title"


def test_injector_get_pdf_title_no_title() -> None:
    with PosixPath("tests/resources/no_title.pdf").open("rb") as file:
        assert Injector(get_config(test=True)).get_pdf_title(file) is None


def test_directory_injector_file_reader() -> None:
    pdf = PosixPath("tests/resources/with_title.pdf")
    assert DirectoryInjector(get_config(test=True)).file_reader(pdf) == Document(
        url="https://invalid/tests/resources/with_title.pdf",
        source="unknown",
        content="\x0c",
        title="Title",
        modtime=pdf.stat().st_mtime,
        size=853,
    )

    odt = PosixPath("tests/resources/document.odt")
    assert DirectoryInjector(get_config(test=True)).file_reader(odt) == Document(
        url="https://invalid/tests/resources/document.odt",
        source="unknown",
        content="Title\n\nContent\n",
        title="Document ODT : document.odt",
        modtime=odt.stat().st_mtime,
        size=8947,
    )

    docx = PosixPath("tests/resources/document.docx")
    assert DirectoryInjector(get_config(test=True)).file_reader(docx) == Document(
        url="https://invalid/tests/resources/document.docx",
        source="unknown",
        content="Content\n",
        title="Document DOCX : document.docx",
        modtime=docx.stat().st_mtime,
        size=9733,
    )

    rtf = PosixPath("tests/resources/document.rtf")
    assert DirectoryInjector(get_config(test=True)).file_reader(rtf) == Document(
        url="https://invalid/tests/resources/document.rtf",
        source="unknown",
        content="Title\n\nContent\n",
        title="Document RTF : document.rtf",
        modtime=rtf.stat().st_mtime,
        size=288,
    )

    epub = PosixPath("tests/resources/document.epub")
    assert DirectoryInjector(get_config(test=True)).file_reader(epub) == Document(
        url="https://invalid/tests/resources/document.epub",
        source="unknown",
        content="Title\n\nContent\n",
        title="Document EPUB : document.epub",
        modtime=epub.stat().st_mtime,
        size=4354,
    )

    html = PosixPath("tests/resources/document.html")
    assert DirectoryInjector(get_config(test=True)).file_reader(html) == Document(
        url="https://invalid/tests/resources/document.html",
        source="unknown",
        content="Title\n\nContent\n",
        title="Document HTML : document.html",
        modtime=html.stat().st_mtime,
        size=3485,
    )

    md = PosixPath("tests/resources/document.md")
    assert DirectoryInjector(get_config(test=True)).file_reader(md) == Document(
        url="https://invalid/tests/resources/document.md",
        source="unknown",
        content="Content\n",
        title="Document MD : document.md",
        modtime=md.stat().st_mtime,
        size=30,
    )
