from json import dumps
from urllib.parse import quote

from flask import Flask

from tests.conftest import database_results_found, database_results_unfound

URL = "https://docs.example/path/to/the/dokuwiki/page"
URL_NONEXISTENT = "https://docs.example/path/to/actually/nothing"


def test_api_patch_nothing(
    app: Flask,
) -> None:
    assert app.test_client().patch("/doc").text == dumps({"msg": "Nothing to commit."})


def test_api_get_doc(
    app: Flask,
) -> None:
    test_api_post_doc(app)

    assert (
        app.test_client()
        .get("/doc?sort=modtime-cache-descending&source=dokuwiki_pages")
        .text
        == "1753797829.0"
    )
    assert app.test_client().get("/doc").status == "501 NOT IMPLEMENTED"
    assert (
        app.test_client().get("/doc?sort=modtime-cache-descending").status
        == "400 BAD REQUEST"
    )


def test_api_post_doc(
    app: Flask,
) -> None:
    database_results_unfound(app)

    response = app.test_client().post(
        "/doc",
        json={
            "source": "dokuwiki_pages",
            "title": "Test title for a DokuWiki page",
            "content": "Here is a testing DokuWiki page content.\nAs this is not a real page, you probably won't learn anything useful here.",
            "url": URL,
            "modtime": 1753797829,
            "size": 115,
        },
    )
    assert response.status == "200 OK"

    assert app.test_client().patch("/doc").text == dumps(
        {"msg": "Modification committed."}
    )

    database_results_found(app)


def test_api_delete_doc(
    app: Flask,
) -> None:
    test_api_post_doc(app)

    database_results_found(app)

    assert app.test_client().delete("/doc/" + quote(URL, safe="")).status == "200 OK"

    assert app.test_client().patch("/doc").text == dumps(
        {"msg": "Modification committed."}
    )

    database_results_unfound(app)


def test_api_delete_doc_unknown(
    app: Flask,
) -> None:
    assert (
        app.test_client().delete("/doc/" + quote(URL_NONEXISTENT, safe="")).status
        == "404 NOT FOUND"
    )
