import os
from shutil import which

from invoke import context, task  # type: ignore[attr-defined]

env = {
    "PYTHONDEVMODE": "1",
}

CI_MODE = bool(os.getenv("CI", default=""))


@task
def fmt(ctx: context.Context) -> None:
    if CI_MODE:
        ctx.run("isort . --verbose", pty=True, env=env)
        ctx.run("black . --verbose", pty=True, env=env)
        ctx.run("git diff --exit-code", pty=True, env=env)
        return
    ctx.run("isort . --quiet", pty=True, env=env)
    ctx.run("black . --quiet", pty=True, env=env)


@task
def yamllint(ctx: context.Context) -> None:
    ctx.run("yamllint . --strict", pty=True, env=env, warn=not CI_MODE)


@task
def taplo(ctx: context.Context) -> None:
    if which("taplo") is not None:
        ctx.run("RUST_LOG=warn taplo format", pty=True, env=env, warn=not CI_MODE)


@task
def ruff(ctx: context.Context) -> None:
    if which("ruff") is not None:
        ctx.run("ruff check --quiet", pty=True, env=env, warn=not CI_MODE)


@task
def flake8(ctx: context.Context) -> None:
    ctx.run("flake8", pty=True, env=env, warn=not CI_MODE)


@task
def pytest(ctx: context.Context) -> None:
    if CI_MODE:
        ctx.run(
            "pytest --cov-report xml:coverage-reports/coverage-pytest.xml",
            pty=True,
            env=env,
        )
    ctx.run("pytest --quiet", pty=True, env=env, warn=True)


@task
def pylint(ctx: context.Context) -> None:
    ctx.run("pylint src", pty=True, env=env, warn=not CI_MODE)


@task
def mypy(ctx: context.Context) -> None:
    ctx.run("mypy .", pty=True, env=env)


@task(fmt, yamllint, taplo, ruff, flake8, pytest, pylint, mypy, default=True)
def check(_: context.Context) -> None:
    pass
