# pylint: disable=no-value-for-parameter

from logging import DEBUG, INFO, WARNING, basicConfig, getLogger
from pathlib import PosixPath

from click import Choice, IntRange, argument, group, option

from eego import get_config
from eego.injector import DirectoryInjector, DokuwikiPagesInjector, RtInjector

basicConfig(
    format="%(levelname)-8.8s [%(asctime)s] %(name)s (%(module)s.py l.%(lineno)s) %(message)s",
)

log = getLogger()


@group()
@option("-v", "--verbose", count=True, type=IntRange(0, 2))
def cli(verbose: int) -> None:
    """Command-line interface for EEGO."""

    match verbose:
        case 0:
            log.setLevel(WARNING)
        case 1:
            log.setLevel(INFO)
        case 2:
            log.setLevel(DEBUG)


@cli.command(context_settings={"max_content_width": 1024})
@argument(
    "source",
    metavar="SOURCE",
    type=Choice(["rt", "rose", "dokuwiki_pages", "dokuwiki_media"]),
)
def inject(source: str) -> None:
    """Inject content from SOURCE."""
    config = get_config()
    if source == "rt":
        RtInjector(config)
    elif source == "dokuwiki_pages":
        DokuwikiPagesInjector(config)
    elif source in ["dokuwiki_media", "rose"]:
        DirectoryInjector(
            config,
            source=source,
            base_path=PosixPath(config["injectors"][source]["base-path"]),
            base_uri=config["injectors"][source]["base-uri"],
        )


if __name__ == "__main__":
    cli()
